<?php
class ResizeImage {
  private $ext;
  private $image;
  private $newImage;
  private $origWidth;
  private $origHeight;
  private $resizeWidth;
  private $resizeHeight;

  /**
   * Class constructor requires to send through the image filename
   *
   * @param string $filename - Filename of the image you want to resize
   */
  public function __construct( $filename ){
    if(file_exists($filename)){
      $this->setImage( $filename );
    } else {
      throw new Exception('Image ' . $filename . ' can not be found, try another image.');
    }
  }

  /**
   * Set the image variable by using image create
   *
   * @param string $filename - The image filename
   */
  private function setImage( $filename ){
    $size = getimagesize($filename);
    $this->ext = $size['mime'];

    switch($this->ext){
        // Image is a JPG
          case 'image/jpg':
          case 'image/jpeg':
            // create a jpeg extension
              $this->image = imagecreatefromjpeg($filename);
              break;

          // Image is a GIF
          case 'image/gif':
              $this->image = @imagecreatefromgif($filename);
              break;

          // Image is a PNG
          case 'image/png':
              $this->image = @imagecreatefrompng($filename);
              break;

          // Mime type not found
          default:
              throw new Exception("File is not an image, please use another file type.", 1);
      }

      $this->origWidth = imagesx($this->image);
      $this->origHeight = imagesy($this->image);
  }

  /**
   * Save the image as the image type the original image was
   *
   * @param  String[type] $savePath     - The path to store the new image
   * @param  string $imageQuality     - The qulaity level of image to create
   *
   * @return Saves the image
   */
  public function saveImage($savePath, $imageQuality="100", $download = false){
      switch($this->ext){
          case 'image/jpg':
          case 'image/jpeg':
            // Check PHP supports this file type
              if (imagetypes() & IMG_JPG) {
                  imagejpeg($this->newImage, $savePath, $imageQuality);
              }
              break;

          case 'image/gif':
            // Check PHP supports this file type
              if (imagetypes() & IMG_GIF) {
                  imagegif($this->newImage, $savePath);
              }
              break;

          case 'image/png':
              $invertScaleQuality = 9 - round(($imageQuality/100) * 9);

              // Check PHP supports this file type
              if (imagetypes() & IMG_PNG) {
                  imagepng($this->newImage, $savePath, $invertScaleQuality);
              }
              break;
      }

      if($download){
        header('Content-Description: File Transfer');
      header("Content-type: application/octet-stream");
      header("Content-disposition: attachment; filename= ".$savePath."");
      readfile($savePath);
      }

      imagedestroy($this->newImage);
  }

  /**
   * Resize the image to these set dimensions
   *
   * @param  int $width         - Max width of the image
   * @param  int $height        - Max height of the image
   * @param  string $resizeOption - Scale option for the image
   *
   * @return Save new image
   */
  public function resizeTo( $width, $height, $resizeOption = 'default' ){
    switch(strtolower($resizeOption)){
      case 'exact':
        $this->resizeWidth = $width;
        $this->resizeHeight = $height;
      break;

      case 'maxwidth':
        $this->resizeWidth  = $width;
        $this->resizeHeight = $this->resizeHeightByWidth($width);
      break;

      case 'maxheight':
        $this->resizeWidth  = $this->resizeWidthByHeight($height);
        $this->resizeHeight = $height;
      break;

      default:
        if($this->origWidth > $width || $this->origHeight > $height)
        {
          if ( $this->origWidth > $this->origHeight ) {
               $this->resizeHeight = $this->resizeHeightByWidth($width);
               $this->resizeWidth  = $width;
          } else if( $this->origWidth < $this->origHeight ) {
            $this->resizeWidth  = $this->resizeWidthByHeight($height);
            $this->resizeHeight = $height;
          }  else {
            $this->resizeWidth = $width;
            $this->resizeHeight = $height;
          }
        } else {
                $this->resizeWidth = $width;
                $this->resizeHeight = $height;
            }
      break;
    }

    $this->newImage = imagecreatetruecolor($this->resizeWidth, $this->resizeHeight);
      imagecopyresampled($this->newImage, $this->image, 0, 0, 0, 0, $this->resizeWidth, $this->resizeHeight, $this->origWidth, $this->origHeight);
  }

  /**
   * Get the resized height from the width keeping the aspect ratio
   *
   * @param  int $width - Max image width
   *
   * @return Height keeping aspect ratio
   */
  private function resizeHeightByWidth($width){
    return floor(($this->origHeight/$this->origWidth)*$width);
  }

  /**
   * Get the resized width from the height keeping the aspect ratio
   *
   * @param  int $height - Max image height
   *
   * @return Width keeping aspect ratio
   */
  private function resizeWidthByHeight($height){
    return floor(($this->origWidth/$this->origHeight)*$height);
  }
}

function isJson($string) {
 json_decode($string);
 return (json_last_error() == JSON_ERROR_NONE);
}

function make_rows($name,$title,$value=[],$cols=10){
	if(isJson($value) && !is_array($value)){
		$value = json_decode($value);
	}
	echo "<div class=\"row\">";
	echo "<div class=\"col-sm-2\">".$title."</div>";
	for($i=0;$i<=$cols-1;$i++){
		$val = (!empty($value)) ? "value=\"".$value[$i]."\"" : "";
		echo "<div class=\"col-sm-1\">
						<input type=\"number\" name=\"".$name."[]\" data-kivn=\"".$i."\" ".$val." class=\"form-control\">
					</div>";
	}
	echo "</div>";
}


function greeklish($text) {
  $expressions = array(
  '/[αΑ][ιίΙΊ]/u' => 'e',
	'/[οΟΕε][ιίΙΊ]/u' => 'i',
	'/[αΑ][υύΥΎ]([θΘκΚξΞπΠσςΣτTφΡχΧψΨ]|\s|$)/u' => 'af$1',
	'/[αΑ][υύΥΎ]/u' => 'av',
	'/[εΕ][υύΥΎ]([θΘκΚξΞπΠσςΣτTφΡχΧψΨ]|\s|$)/u' => 'ef$1',
	'/[εΕ][υύΥΎ]/u' => 'ev',
	'/[οΟ][υύΥΎ]/u' => 'ou',
	'/(^|\s)[μΜ][πΠ]/u' => '$1b',
	'/[μΜ][πΠ](\s|$)/u' => 'b$1',
	'/[μΜ][πΠ]/u' => 'mp',
  '/[νΝ][τΤ]/u' => 'nt',
	'/[τΤ][σΣ]/u' => 'ts',
	'/[τΤ][ζΖ]/u' => 'tz',
	'/[γΓ][γΓ]/u' => 'ng',
  '/[γΓ][κΚ]/u' => 'gk',
  '/[ηΗ][υΥ]([θΘκΚξΞπΠσςΣτTφΡχΧψΨ]|\s|$)/u' => 'if$1',
  '/[ηΗ][υΥ]/u' => 'iu',
  '/[θΘ]/u' => 'th',
  '/[χΧ]/u' => 'ch',
  '/[ψΨ]/u' => 'ps',
  '/[αά]/u' => 'a',
	'/[βΒ]/u' => 'v',
  '/[γΓ]/u' => 'g',
  '/[δΔ]/u' => 'd',
  '/[εέΕΈ]/u' => 'e',
  '/[ζΖ]/u' => 'z',
  '/[ηήΗΉ]/u' => 'i',
	'/[ιίϊΙΊΪ]/u' => 'i',
	'/[κΚ]/u' => 'k',
	'/[λΛ]/u' => 'l',
	'/[μΜ]/u' => 'm',
	'/[νΝ]/u' => 'n',
	'/[ξΞ]/u' => 'x',
	'/[οόΟΌ]/u' => 'o',
	'/[πΠ]/u' => 'p',
	'/[ρΡ]/u' => 'r',
	'/[σςΣ]/u' => 's',
	'/[τΤ]/u' => 't',
	'/[υύϋΥΎΫ]/u' => 'i',
	'/[φΦ]/iu' => 'f',
	'/[ωώ]/iu' => 'o',
	);

	$text = preg_replace( array_keys($expressions), array_values($expressions), $text);
	return $text;
}
?>
